Function LoadOrCreateMetadata(videoMetadata, videoPlayerKey)
    metadata = videoMetadata.Lookup(videoPlayerKey)
    if metadata = Invalid then
        return CreateObject("roAssociativeArray")
    endif
    return metadata
End Function

Sub SetMetadataType(videoMetadata, videoPlayerKey, metadataType)
    metadata = LoadOrCreateMetadata(videoMetadata, videoPlayerKey)
    metadata["type"] = metadataType
    videoMetadata.AddReplace(videoPlayerKey, metadata)
End Sub

Sub SetVideoMetadataEnded(videoMetadata, videoPlayerKey, ended)
    ' used for flaging video hit time code for the first time
    metadata = LoadOrCreateMetadata(videoMetadata, videoPlayerKey)
    metadata["ended"] = ended
    videoMetadata.AddReplace(videoPlayerKey, metadata)
End Sub

Sub SetVideoMetadata(videoPlayers, videoMetadata, videoPlayerKey, isLooped)
    metadata = LoadOrCreateMetadata(videoMetadata, videoPlayerKey)
    if metadata["type"] = "video" then
        videoPlayer = videoPlayers.Lookup(videoPlayerKey)
        metadata["timer"] = CreateObject("roTimeSpan")
        metadata["duration"] = videoPlayer.GetDuration()
        metadata["stopped"] = false
        metadata["looped"] = isLooped
        videoMetadata.AddReplace(videoPlayerKey, metadata)
    endif
End Sub

Sub SetMetadataInvocationUid(videoMetadata, videoPlayerKey, isLooped, invocationUid)
    metadata = LoadOrCreateMetadata(videoMetadata, videoPlayerKey)
    metadata["invocationUid"] = invocationUid
    metadata["looped"] = isLooped
    videoMetadata.AddReplace(videoPlayerKey, metadata)
End Sub

Sub InvokeJSMessage(htmlWidget, invocationUid, result, response)
    if response = invalid then
        response = CreateObject("roAssociativeArray")
    end if
    response["type"] = "response"
    response["succeeded"] = result
    response["invocationUid"] = invocationUid
    htmlWidget.PostJSMessage(response)
    response = invalid
End Sub

Sub SetMetadataStopped(videoMetadata, videoPlayerKey)
    metadata = LoadOrCreateMetadata(videoMetadata, videoPlayerKey)
    metadata["stopped"] = true
    metadata["looped"] = false
    metadata["ended"] = false
    videoMetadata.AddReplace(videoPlayerKey, metadata)
End Sub

Sub AppendEndedEvent(videoPlayer)
    videoDuration = videoPlayer.GetDuration()
    videoPosition = videoPlayer.GetPlaybackPosition()
    remainingTime = videoDuration - videoPosition
    ' start the progressive video end tracking 1.5 second before the video ends
    videoPlayer.AddEvent(8, remainingTime - 1500)
End Sub

Sub PrepareMedia(screenRegistrySection, videoModeProvider, videoPlayer, event, videoFilePath, canVideosBePreloaded)
    x = strtoi(event.message.x)
    y = strtoi(event.message.y)
    width = strtoi(event.message.width)
    height = strtoi(event.message.height)
    orientation = GetCurrentVideoTransform(screenRegistrySection)
    rectangle = CreateObject("roRectangle", x, y, width, height)
    videoPlayer.SetRectangle(rectangle)
    videoPlayer.SetTransform(orientation)

    ' Loop mode needs to be set before video prepare otherwise loop mode won't work even
    ' if "GetPropertiesString" method will return right value
    ' We use LoopButNotSeamless to avoid video freezes by looping FHD/UHD/HDREADY videos
    videoPlayer.SetLoopMode("LoopButNotSeamless")

    if event.message.background = "true" then
        videoModeProvider.SetGraphicsZOrder("front")
    else
        videoModeProvider.SetGraphicsZOrder("back")
    endif

    if canVideosBePreloaded = true then
        videoPlayer.PreloadFile({ filename: videoFilePath })
    end if
End Sub

Function CanPreloadVideos(videoPlayers, deviceFamily)
    ' Special handling for Series 5 as they have issues with video layering
    if deviceFamily = "cobra" then
        return false
    else if deviceFamily = "raptor" then
        return false
    end if

    ' hack for speed, if there is secondary video player apart from video_0, we can preload videos in advance
    return videoPlayers.DoesExist("video_1")
End Function

Function PlayVideo(videoPlayer, filePath, volume, deviceFamily, canVideosBePreloaded)
    videoStartedPlayback = true
    ' TEMP hotfix to enable video playback on LG UV5N
    ' Until there is a fix in the FW for video layers
    ' Broken in FW 9.1.32
    if deviceFamily = "thor" then
        videoPlayConfig = CreateObject("roAssociativeArray")
        videoPlayConfig.filename = filePath
        videoPlayConfig.zindex = 21
        videoStartedPlayback = videoPlayer.PlayFile(videoPlayConfig)
        videoPlayConfig = invalid
    else
        if canVideosBePreloaded = true then
            ' playing preloaded video
            videoStartedPlayback = videoPlayer.Play()
        else
            ' playing video without preload
            videoStartedPlayback = videoPlayer.PlayFile(filePath)
        end if
    end if

    return videoStartedPlayback
End Function

Sub PrepareStream(videoPlayer, uri)
    stream = CreateObject("roRtspStream", uri)

    print "PreloadFile:Stream"

    videoPlayer.PreloadFile({ rtsp: stream })
End Sub

Sub PlayStream(videoPlayer, uri)
    stream = CreateObject("roRtspStream", uri)
    videoPlayer.PlayFile({ rtsp: stream })
End Sub

Sub PlayHDMI(videoPlayer)
    videoInput = CreateObject("roVideoInput")
    videoPlayer.PlayFile(videoInput)
End Sub

Sub InitVideoMode(screenRegistrySection, supportedVideoModes, defaultWidth, defaultHeight)
    if not(screenRegistrySection.Exists("video_mode")) then
        SaveVideoModeIfSupported(screenRegistrySection, supportedVideoModes, defaultWidth, defaultHeight, "auto")
    endif
End Sub

Function SaveVideoModeIfSupported(screenRegistrySection, supportedVideoModes, width, height, framerate)
    supportedVideoModes.Reset()
    While supportedVideoModes.IsNext()
        mode = supportedVideoModes.Next()
        if (framerate = "auto" OR mode["framerate"] = strtoi(framerate)) AND mode["width"] = strtoi(width) AND mode["height"] = strtoi(height) then
            screenRegistrySection.Write("video_mode", FormatJson(mode, 1))
            screenRegistrySection.Flush()
            return true
        endif
    End While
    return false
End Function

Function Supports4kWidget() As Boolean
    deviceInfo = CreateObject("roDeviceInfo")
    model = deviceInfo.GetModel()

    ' Check if current model is in the supported list
    if model = "XT1144" or model = "XT1144-t" or model = "XT244" or model = "XT1144-pp" or model = "XT245" or model = "XT1145" or model = "XC4055" or model = "XT2055" or model = "XT2145" then
        return true
    endif

    return false
End Function

Sub ApplySavedVideoMode(screenRegistrySection, videoModeProvider, fullScreenRectangle)
    if screenRegistrySection.Exists("video_mode") then
        videoMode = ParseJson(screenRegistrySection.Read("video_mode"))
        if Supports4kWidget() = true then
            width = videoMode["width"]
            height = videoMode["height"]
        else
            width = videoMode["graphicsPlaneWidth"]
            height = videoMode["graphicsPlaneHeight"]
        end if
        fullScreenRectangle.SetWidth(width)
        fullScreenRectangle.SetHeight(height)
        if width > 1920 then
            videoModeProvider.SetMode(videoMode["videomode"] + ":fullres")
        else
            videoModeProvider.SetMode(videoMode["videomode"])
        end if
        print "videoMode=";screenRegistrySection.Read("video_mode")
    endif

    if screenRegistrySection.Exists("orientation") then
        print "orientation=";screenRegistrySection.Read("orientation")
    endif

    if screenRegistrySection.Exists("video_orientation") then
        print "video orientation=";screenRegistrySection.Read("video_orientation")
    endif
End Sub

Function GetCurrentVideoTransform(screenRegistrySection)
    if screenRegistrySection.Exists("video_orientation") then
        return "identity" ' LANDSCAPE_FLIPPED is not supported on this platform
    endif
    return GetCurrentTransform(screenRegistrySection)
End Function

' Backward functionality with filesystem integration
Function GetFallbackPath(path as String)
    if path = "SD:" then
        return "/storage/sd"
    endif
    if path = "SSD:" then
        return "/storage/ssd"
    endif
    if path = "USB1:" then
        return "/storage/usb1"
    endif
    if path = "FLASH:" then
        return "/storage/flash"
    endif
    return "/storage/sd" 'default
End Function

Sub EnableTouchScreen(msgPort, screenRegistrySection)
    if screenRegistrySection.Exists("video_mode") then
        touchVideoMode = ParseJson(screenRegistrySection.Read("video_mode"))
        touchWidth = touchVideoMode["graphicsPlaneWidth"]
        touchHeight = touchVideoMode["graphicsPlaneHeight"]
        touchScreen = CreateObject("roTouchScreen")
        touchScreen.SetPort(msgPort)
        touchScreen.SetResolution(touchWidth, touchHeight)
    endif
    print "EnableTouchScreen completed"
End Sub

Sub SetupOnScreenKeyboard(msgPort, screenRegistrySection, onScreenKeyboardObject)
    if IsOnScreenKeyboardSupported() then
        onScreenKeyboardObject.Clear()
        print "SOS SetupOSK"
        currentTransform = GetCurrentTransform(screenRegistrySection)
        ' This is ready for the moment BrightSign fixes the events for virtual keyboard
        ' Until then we have to use fullscreen keyboard rectangle
        'if currentTransform = "identity" then
        '    rv=createObject("roRectangle", 480,746,960,334)
        'else if currentTransform = "rot90"
        '    rv=createObject("roRectangle", 1586,0,334,1080)
        'else if currentTransform = "rot270" then
        '    rv=createObject("roRectangle", 0,0,334,1080)
        'else
        '    rv=createObject("roRectangle", 0,0,1920,1080)
        'endif

        rv=createObject("roRectangle", 0,0,1920,1080)
        osk=createObject("roVirtualKeyboard",rv)
        osk.setTransform(currentTransform)
        osk.setResource("file:///osk/bsvirtualkb.html")
        osk.SetPort(msgPort)
        onScreenKeyboardObject.AddReplace("onScreenKeyboard", osk)
    end if
End Sub

Function IsOnScreenKeyboardSupported() As Boolean
    deviceInfo = CreateObject("roDeviceInfo")
    model = deviceInfo.GetModel()
    xtPPRegex = CreateObject("roRegex", "^xt1144-pp$", "i")

    if xtPPRegex.IsMatch(model) then
        currentVersion = deviceInfo.GetVersion()
        ' List of supported firmware versions for the on-screen keyboard
        supportedVersions = ["8.5.53", "8.5.53.2", "8.5.60", "8.5.64"]
        for each version in supportedVersions
            if currentVersion = version then
                return true
            end if
        end for
    end if

    return false
End Function

Function SetupVideoCanvasAndResolution(videoModeProvider, screenRegistrySection, supportedVideoModes)
    deviceInfo = CreateObject("roDeviceInfo")
    model = deviceInfo.GetModel()
    print "model=";model
    hsRegex = CreateObject("roRegex", "^hs\w+$", "i")
    xtPPRegex = CreateObject("roRegex", "^xt1144-pp$", "i")
    xdPPRegex = CreateObject("roRegex", "^xd1034-w$", "i")
    lgRegex = CreateObject("roRegex", "^lg\w+$", "i")
    if hsRegex.IsMatch(model) then
        print "HSxxxx series"
        defaultResolution = { width: 1280, height: 800 }
        InitVideoMode(screenRegistrySection, supportedVideoModes, "1280", "800")
    else if xtPPRegex.IsMatch(model) then
        print "XT1144-PP series"
        defaultResolution = { width: 1920, height: 1080 }
        InitVideoMode(screenRegistrySection, supportedVideoModes, "1920", "1080")
    else if xdPPRegex.IsMatch(model) then
        print "XD1034-W series"
        defaultResolution = { width: 1920, height: 1080 }
        InitVideoMode(screenRegistrySection, supportedVideoModes, "1920", "1080")
    else if lgRegex.IsMatch(model) then
        print "LG UV5N series"
        defaultResolution = { width: 1920, height: 1080 }
        InitVideoMode(screenRegistrySection, supportedVideoModes, "1920", "1080")
    else
        print "Any specific model hasn't been recognized"
        defaultResolution = { width: 800, height: 600 }
        InitVideoMode(screenRegistrySection, supportedVideoModes, "800", "600")
    endif

    fullScreenRectangle = CreateObject("roRectangle", 0, 0, defaultResolution.width, defaultResolution.height)
    ApplySavedVideoMode(screenRegistrySection, videoModeProvider, fullScreenRectangle)

    print "SetupVideoCanvasAndResolution completed"
    return fullScreenRectangle
End Function

Sub FirstRunSetup(systemTime)

    firstStartSection = CreateObject("roRegistrySection", "firstStart")
    if not(firstStartSection.Exists("time")) then
        print "This is first start"
        currentLocalTime = systemTime.GetLocalDateTime()
        firstStartSection.Write("time", currentLocalTime.GetString())
        ' Hook to custom.brs
        OnFirstStartHook()
    endif

    print "First setup completed"
End Sub

Function FormatDNSServersArray(dnsServersArray)
    ' Initialize an empty roArray object to hold the DNS server IPs
    dnsArray = CreateObject("roArray", 0, true)

    ' transform roAssocArray to roArray supported by SetDNSServers()
    for each key in dnsServersArray
        dnsServer = dnsServersArray.Lookup(key)
        if dnsServer <> invalid then
            dnsArray.Push(dnsServer)
        end if
    end for

    ' Check if the number of DNS servers exceeds the allowed limit
    if dnsArray.Count() > 3 then
        print "Error: More than 3 DNS servers are not allowed."
        return invalid
    end if

    ' Return the formatted DNS array
    return dnsArray
End Function

Function GetBrowserWidgetSecuritySettings(parameter)
    browserSecurityRegistrySection = CreateObject("roRegistrySection", "browserSecurity")

    if browserSecurityRegistrySection.Read(parameter) = "enabled" then
        return true
    else if browserSecurityRegistrySection.Read(parameter) = "disabled" then
        return false
    else
        ' default values
        if parameter = "websecurity" then
            return false
        end if

        if parameter = "camera_enabled" then
            return true
        end if

        if parameter = "insecure_https_enabled" then
            return true
        end if

    end if

    return false

End Function

Function SetupBrowserWidget(msgPort, screenRegistrySection, htmlPath, x, y, width, height, browserWindows)

    currentTransform = GetCurrentTransform(screenRegistrySection)
    firstAvaiableStorage = GetFirstAvaliableStorageUnit()
    hwzSettings = GetCurrentHWZSettings()

    websecurityFlag = GetBrowserWidgetSecuritySettings("websecurity")
    cameraEnabledFlag = GetBrowserWidgetSecuritySettings("camera_enabled")
    insecureHttpsFlag = GetBrowserWidgetSecuritySettings("insecure_https_enabled")

    config = {
        nodejs_enabled: true,
        brightsign_js_objects_enabled: true,
        mouse_enabled: true,
        focus_enabled: true,
        javascript_enabled: true,
        url: htmlPath,
        user_stylesheet: firstAvaiableStorage + "/widget.css",
        transform: currentTransform,
        hwz_default: hwzSettings,
        pinch_to_zoom_enabled: true,
        scrollbar_enabled: true,
        security_params: {
            websecurity: websecurityFlag,
            camera_enabled: cameraEnabledFlag,
            insecure_https_enabled: insecureHttpsFlag
        }
    }
    ' tmp disabled, probably bug in FW
    

    screenRectangle = CreateObject("roRectangle", strtoi(x), strtoi(y), strtoi(width), strtoi(height))

    ' using htmlWidget to limit possible conflict with 3rd party htmlWidget created in custom.brs
    browserWidget = CreateObject("roHtmlWidget", screenRectangle, config)
    browserWidget.SetPort(msgPort)

    proxyServer = GetProxyServer()
    proxyServerBypass = GetProxyBypass()

    browserWidget.SetProxy(proxyServer)
    browserWidget.SetProxyBypass(proxyServerBypass)

    browserWidget.Show()
    print "SetupBrowserWidget completed"
    browserWindows.AddReplace("browser", browserWidget)
    browserWindows.AddReplace("browser_x", x)
    browserWindows.AddReplace("browser_y", y)
    browserWindows.AddReplace("browser_width", width)
    browserWindows.AddReplace("browser_height", height)
    return browserWidget
End Function

Function DiscardBrowserWidget(browserWindows)

    browserWidget = browserWindows.Lookup("browser")
    if browserWidget <> invalid then
        browserWidget.Hide()
        browserWidget = invalid

        print "DiscardBrowserWidget completed"
        browserWindows.Delete("browser")
    end if

    return true
End Function

Function IsPdfFileOnUrl(mimeType as String, url as String) as Boolean
    if mimeType = "application/pdf" then
        return true
    else if mimeType = "application/octet-stream" then
        ' Ensure URL is valid and long enough
        if url = invalid or Len(url) < 5 then return false
        ' Convert to lowercase for case-insensitive comparison
        url = LCase(url)

        ' Remove query parameters and fragments
        questionMarkPos = Instr(1, url, "?")
        hashPos = Instr(1, url, "#")

        if questionMarkPos > 0 and hashPos > 0 then
            cutoffPos = Min(questionMarkPos, hashPos) - 1
            url = Left(url, cutoffPos)
        else if questionMarkPos > 0 then
            url = Left(url, questionMarkPos - 1)
        else if hashPos > 0 then
            url = Left(url, hashPos - 1)
        end if

        ' Now check if the cleaned URL ends with ".pdf"
        return Right(url, 4) = ".pdf"
    end if

    return false
End Function
