
' Recovery script that resolves any unexpected application issue
Sub RunRecoveryCheck()
    path$ = RecoveryFindDestPath()
    ' Should return an array of [storageUnit and filename]
    recoveryPackagePath = RecoveryFindSourcePath()

    shouldForceRecover = RecoveryGetFlag()

    if Type(recoveryPackagePath) = "roArray" then
        RecoverySetFlag("0")
        RecoverySignageOSLog(["RECOVERY", "will perform CA installation recovery"])
        RecoveryRunAppInstall(recoveryPackagePath)
    else if shouldForceRecover = "1" then
        RecoverySignageOSLog(["RECOVERY", "will perform forced recovery"])
        RecoveryForceRecovery()
    else
        print "Nothing to recover from"
        RecoverySignageOSLog(["nothing to recover from"])
    end if

End Sub

Sub RecoveryRunAppInstall(recoveryPackagePath)
        storageUnit = recoveryPackagePath[0] + "/"
        fileName = recoveryPackagePath[1]
        fullRecoveryPackagePath = storageUnit + fileName
        tmpRecoveryFolder = storageUnit + Mid(fileName, 1, Len(fileName)-4) + "/"

        RecoverySignageOSLog(["found recovery package", fullRecoveryPackagePath])

        ' add additional sleep time to allow BrightSign to connect to the network and eventually start any cloud-driven recovery
        sleep(15000)

         RecoverySignageOSLog(["path]", tmpRecoveryFolder])

		DeleteDirectory(tmpRecoveryFolder)
		CreateDirectory(tmpRecoveryFolder)


        package = CreateObject("roBrightPackage", fullRecoveryPackagePath)
        if Type(package) = "roBrightPackage" then
            package.Unpack(tmpRecoveryFolder)
            package = 0
            RecoverySignageOSLog(["RECOVERY", "application unzipped"])
        else
            RecoverySignageOSLog(["RECOVERY", "ERROR: Failed to create roBrightPackage, ZIP file may be corrupted"])
            RecoverySignageOSLog(["RECOVERY", "Deleting corrupted recovery package"])
            DeleteFile(fullRecoveryPackagePath)
            DeleteDirectory(tmpRecoveryFolder)
            return
        end if

        files = MatchFiles(tmpRecoveryFolder, "*")
        for each file in files
            if file <> "custom.brs" then
                CopyFile(tmpRecoveryFolder + file, storageUnit + file)
            end if
        end for

        RecoverySignageOSLog(["RECOVERY", "files copied"])

        folders = MatchFiles(tmpRecoveryFolder, "*")
        for each file in folders
            if file <> "custom.brs" then
                CopyFile(tmpRecoveryFolder + file + "/", storageUnit + file + "/")
            end if
        end for

        RecoverySignageOSLog(["RECOVERY", "folders copied"])

        DeleteDirectory(tmpRecoveryFolder)
        MoveFile(fullRecoveryPackagePath, storageUnit + "display-brightsign.backup")

        RecoverySignageOSLog(["RECOVERY", "cleanup completed, rebooting in 5 seconds"])

        RecoveryLogRecoveryAttempt()

        sleep(5000)
        a = RebootSystem()
End Sub

Function GetRecoveryPackage(sourceDir)
    packageRegexName = CreateObject("roRegex", "^display-brightsign", "i")

    results = MatchFiles(sourceDir, "*.zip")
    for each file in results
        if packageRegexName.IsMatch(file) then
            return file
        end if
    end for

    return false
End Function

Function RecoveryFindDestPath()
    if not RecoveryIsFirmwareValid() then
        return "SD:/"
    end if

    destinationPaths = ["SSD:", "SD:", "USB1:"]
    di = CreateObject("roDeviceInfo")
    family = lcase(di.GetFamily())
    if family = "thor" then
        destinationPaths.push("FLASH:")
    end if
    for each destination in destinationPaths
        if RecoveryIsMounted(destination) then
            return destination + "/"
        end if
    next
    return "unknown"
End Function

Function RecoveryFindSourcePath()
    if not RecoveryIsFirmwareValid() then
        return false
    end if

    sourcePaths = ["USB1:", "SD:", "SSD:"]
    di = CreateObject("roDeviceInfo")
    family = lcase(di.GetFamily())
    if family = "thor" then
        sourcePaths.push("FLASH:")
    end if
    for each source in sourcePaths
        if RecoveryIsMounted(source) then
            recoveryPackage = GetRecoveryPackage(source + "/")
            if Type(recoveryPackage) = "roString" then
                return [source, recoveryPackage]
            end if
            recoveryPackage = invalid
        end if
    next

    return false
End Function

Function RecoveryIsMounted(path as String)
    if CreateObject("roStorageHotplug").GetStorageStatus(path).mounted then
        return true
    end if

    return false
End Function

Function RecoveryIsFirmwareValid()
    di = CreateObject("roDeviceInfo")
    return di.FirmwareIsAtLeast("7.0.60")
End Function

' Create necessary objects
Function RecoveryDownloadZipFile() as Boolean
    urlTransfer = CreateObject("roUrlTransfer")
    unzipArchive = CreateObject("roUnzip")

    ' Define paths
    path = RecoveryFindDestPath()
    zipFilePath = path + "display-brightsign.backup"

    ' Step 1: Download ZIP file from https://https://2.signageos.io/app/brightsign/2.3.4/autorun.zip
    ' Note: Replace the URL with the direct link to the ZIP file if necessary
    urlTransfer.SetUrl("https://b.signageos.io/autorun.zip")
    result = urlTransfer.GetToFile(zipFilePath)

    If result <> 200 Then
        errorMsg = urlTransfer.GetFailureReason()
        Print "Error downloading ZIP file. Result code: "; result
        Print "Failure reason: "; errorMsg
        RecoverySignageOSLog(["RECOVERY", "download of application package failed", result, errorMsg])
        return false
    End If

    MoveFile(zipFilePath, path + "display-brightsign-backup.zip" )
    RecoverySetFlag("0")
    return true
End Function

Sub RecoveryUnpackZipFile()
    path = RecoveryFindDestPath()
    package = CreateObject("roBrightPackage", path + "display-brightsign-backup.zip")
    package.Unpack(path)
    MoveFile(path + "display-brightsign-backup.zip", path + "display-brightsign.backup")
    RecoveryLogRecoveryAttempt()
    RebootSystem()
End Sub

Sub RecoveryForceRecovery()
    ' try to recover from local application package backup
    RecoverySignageOSLog(["RECOVERY", "looking for backup application package"])
    path = RecoveryFindDestPath()
    results = MatchFiles(path, "display-brightsign.backup")
    for each file in results
            RecoverySignageOSLog(["RECOVERY", "backup application package found, recovery will continue after reboot"])
            MoveFile(path + file, path + "display-brightsign-backup.zip")
            RecoverySetFlag("0")
            RecoveryLogRecoveryAttempt()
            RebootSystem()
    end for

    ' if there is no local backup, continue with remote recovery
    sleep(10000)

    RecoverySignageOSLog(["RECOVERY", "recovering from cloud, downloading new application package"])

    if RecoveryDownloadZipFile() then
        RecoveryUnpackZipFile()
    else
        ' to avoid boot loop
        RecoverySetFlag("0")
        RecoverySignageOSLog(["RECOVERY", "recovery from cloud failed"])
    end if

    sleep(10000)

    RebootSystem()

End Sub

Sub RecoverySetFlag(flag)
    recoverySection = CreateObject("roRegistrySection", "recovery")
    recoverySection.write("forceRecovery",flag)
    recoverySection.flush()
End Sub

Function RecoveryGetFlag()
    recoverySection = CreateObject("roRegistrySection", "recovery")
    forceRecovery = recoverySection.read("forceRecovery")
    return forceRecovery
End Function

Sub RecoveryLogRecoveryAttempt()
    systemTime = CreateObject("roSystemTime")
    recoverySection = CreateObject("roRegistrySection", "recovery")
    currentLocalTime = systemTime.GetLocalDateTime()
    recoverySection.write("lastRecoveryAttemptTime", currentLocalTime.GetString())
End Sub

Sub RecoverySignageOSLog(logParts)
	logLine = "signageOS "
	systemLog = CreateObject("roSystemLog")
	for each logPart in logParts
		logPartStr = logPart
		if type(logPart) = "roBoolean" then
			logPartStr = signageOS_BooleanToString(logPart)
		end if
		logLine = logLine + logPartStr + " "
	next
	systemLog.SendLine(logLine)
End Sub

Function signageOS_BooleanToString(value as Boolean)
	if value then
		return "TRUE"
	end if
	return "FALSE"
End Function
